//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents the base class from which all implementations of anonymous factories must derive.   
	/// </summary>
	/// <typeparam name="TObjectToCreate">The object type to create.</typeparam>
	/// <typeparam name="TObjectConfiguration">The object configuration type used to inject data.</typeparam>
	/// <remarks>
	/// Anonymous injection factory implements the <see cref="IAnonymousObjectFactory"/> to allow creation of non-named objects through configuration.
	/// </remarks>
	public abstract class AnonymousObjectInjectionFactory<TObjectToCreate, TObjectConfiguration> : InjectionFactory<TObjectToCreate, TObjectConfiguration>, IObjectFactory<TObjectToCreate, object>
	{		
		/// <summary>
		/// Create an instance of a <typeparamref name="TObjectToCreate"/> object.
		/// </summary>
		/// <returns>An instance of a <typeparamref name="TObjectToCreate"/> object.</returns>
		public TObjectToCreate Create()
		{
			return Create(new SystemConfigurationSource());
		}

		/// <summary>
		/// Create an instance of a <typeparamref name="TObjectToCreate"/> object using an <see cref="IConfigurationSource"/>.
		/// </summary>
		/// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
		/// <returns></returns>
		public TObjectToCreate Create(IConfigurationSource configurationSource)
		{
			return base.Create(GetConfiguration(configurationSource), configurationSource);
		}

		/// <summary>
		/// Creates an object.
		/// </summary>
		/// <returns>The object instance.</returns>
        TObjectToCreate IObjectFactory<TObjectToCreate, object>.CreateObject(object unused)
		{
			return Create();
		}

		/// <summary>
		/// Creates an object using the configuration source.
		/// </summary>
		/// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
		/// <returns>The object instance.</returns>
        TObjectToCreate IObjectFactory<TObjectToCreate, object>.CreateObject(object unused, IConfigurationSource configurationSource)
		{
			return Create(configurationSource);
		}
		
		/// <summary>
		/// Get the configuration for the object to create from the configuration source.
		/// </summary>
		/// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
		/// <returns>An instance of <typeparamref name="TObjectConfiguration"/> object.</returns>
		protected abstract TObjectConfiguration GetConfiguration(IConfigurationSource configurationSource);
	}
}
